﻿
#define USE_DYNAMIC_DPAD

using UnityEngine;
using System.Collections;

public class PlayerControl : MonoBehaviour
{
	[HideInInspector]
	public bool facingRight = true;			// For determining which way the player is currently facing.
	[HideInInspector]
	public bool jump = false;				// Condition for whether the player should jump.


	public float moveForce = 365f;			// Amount of force added to move the player left and right.
	public float maxSpeed = 5f;				// The fastest the player can travel in the x axis.
	public AudioClip[] jumpClips;			// Array of clips for when the player jumps.
	public float jumpForce = 1000f;			// Amount of force added when the player jumps.
	public AudioClip[] taunts;				// Array of clips for when the player taunts.
	public float tauntProbability = 50f;	// Chance of a taunt happening.
	public float tauntDelay = 1f;			// Delay for when the taunt should happen.

#if USE_DYNAMIC_DPAD
	public DynamicDpad dynamicDpad;		// Dynamic d-pad
	public GUITexture dpadGui;			// Dpad GUI texture
	public GUITexture dpadKnobGui;		// Dpad knob GUI texture
#endif //USE_DYNAMIC_DPAD


	private int tauntIndex;					// The index of the taunts array indicating the most recent taunt.
	private Transform groundCheck;			// A position marking where to check if the player is grounded.
	private bool grounded = false;			// Whether or not the player is grounded.
	private Animator anim;					// Reference to the player's animator component.


	void Awake()
	{
		// Setting up references.
		groundCheck = transform.Find("groundCheck");
		anim = GetComponent<Animator>();


#if USE_DYNAMIC_DPAD
#if UNITY_EDITOR
		// Make sure dynamic dpad and dpad GUIS are Not the same object, otherwise the dynamic dpad will be disabled when the GUIS are hidden
		if (dynamicDpad != null)
		{
			if ((dpadGui != null) && (dpadGui.gameObject == dynamicDpad.gameObject))
			{
				Debug.LogError("ERROR: Dynamic Dpad component must not be on the same game object as the Dpad GUI texture.");
			}
			if ((dpadKnobGui != null) && (dpadKnobGui.gameObject == dynamicDpad.gameObject))
			{
				Debug.LogError("ERROR: Dynamic Dpad component must not be on the same game object as the Dpad Knob GUI texture.");
			}
		}
#endif //UNITY_EDITOR
		
		if (dpadGui != null)
		{
			dpadGui.gameObject.SetActive(false);
		}
		if (dpadKnobGui != null)
		{
			dpadKnobGui.gameObject.SetActive(false);
		}
#endif //USE_DYNAMIC_DPAD

	}


	void Update()
	{
		// The player is grounded if a linecast to the groundcheck position hits anything on the ground layer.
		grounded = Physics2D.Linecast(transform.position, groundCheck.position, 1 << LayerMask.NameToLayer("Ground"));  

		// If the jump button is pressed and the player is grounded then the player should jump.
		if(Input.GetButtonDown("Jump") && grounded)
			jump = true;


#if USE_DYNAMIC_DPAD
		// Set GUI positions
		if (dynamicDpad != null)
		{
			// Only render when there is a touch
			bool visible = dynamicDpad.HasTouch;
			
			if (dpadGui != null)
			{
				if (dpadGui.gameObject.activeInHierarchy != visible)
				{
					dpadGui.gameObject.SetActive(visible);
				}
				if (visible)
				{
					dynamicDpad.PositionGUITexture(dpadGui);
				}
			}
			
			if (dpadKnobGui != null)
			{
				if (dpadKnobGui.gameObject.activeInHierarchy != visible)
				{
					dpadKnobGui.gameObject.SetActive(visible);
				}
				if (visible)
				{
					dynamicDpad.PositionKnobGUITexture(dpadKnobGui);
				}
			}
		}
#endif //USE_DYNAMIC_DPAD

	}


	void FixedUpdate ()
	{
		// Cache the horizontal input.
		float h = Input.GetAxis("Horizontal");

#if USE_DYNAMIC_DPAD
		if (dynamicDpad != null)
		{
			h = dynamicDpad.X;
		}
#endif //USE_DYNAMIC_DPAD

		// The Speed animator parameter is set to the absolute value of the horizontal input.
		anim.SetFloat("Speed", Mathf.Abs(h));

		// If the player is changing direction (h has a different sign to velocity.x) or hasn't reached maxSpeed yet...
		if(h * rigidbody2D.velocity.x < maxSpeed)
			// ... add a force to the player.
			rigidbody2D.AddForce(Vector2.right * h * moveForce);

		// If the player's horizontal velocity is greater than the maxSpeed...
		if(Mathf.Abs(rigidbody2D.velocity.x) > maxSpeed)
			// ... set the player's velocity to the maxSpeed in the x axis.
			rigidbody2D.velocity = new Vector2(Mathf.Sign(rigidbody2D.velocity.x) * maxSpeed, rigidbody2D.velocity.y);

		// If the input is moving the player right and the player is facing left...
		if(h > 0 && !facingRight)
			// ... flip the player.
			Flip();
		// Otherwise if the input is moving the player left and the player is facing right...
		else if(h < 0 && facingRight)
			// ... flip the player.
			Flip();

		// If the player should jump...
		if(jump)
		{
			// Set the Jump animator trigger parameter.
			anim.SetTrigger("Jump");

			// Play a random jump audio clip.
			int i = Random.Range(0, jumpClips.Length);
			AudioSource.PlayClipAtPoint(jumpClips[i], transform.position);

			// Add a vertical force to the player.
			rigidbody2D.AddForce(new Vector2(0f, jumpForce));

			// Make sure the player can't jump again until the jump conditions from Update are satisfied.
			jump = false;
		}
	}
	
	
	void Flip ()
	{
		// Switch the way the player is labelled as facing.
		facingRight = !facingRight;

		// Multiply the player's x local scale by -1.
		Vector3 theScale = transform.localScale;
		theScale.x *= -1;
		transform.localScale = theScale;
	}


	public IEnumerator Taunt()
	{
		// Check the random chance of taunting.
		float tauntChance = Random.Range(0f, 100f);
		if(tauntChance > tauntProbability)
		{
			// Wait for tauntDelay number of seconds.
			yield return new WaitForSeconds(tauntDelay);

			// If there is no clip currently playing.
			if(!audio.isPlaying)
			{
				// Choose a random, but different taunt.
				tauntIndex = TauntRandom();

				// Play the new taunt.
				audio.clip = taunts[tauntIndex];
				audio.Play();
			}
		}
	}


	int TauntRandom()
	{
		// Choose a random index of the taunts array.
		int i = Random.Range(0, taunts.Length);

		// If it's the same as the previous taunt...
		if(i == tauntIndex)
			// ... try another random taunt.
			return TauntRandom();
		else
			// Otherwise return this index.
			return i;
	}
}
