﻿using UnityEngine;
using System.Collections;

/// <summary>
/// Demo to show how the calibration works.
/// </summary>
public class Demo : MonoBehaviour {

	// Const/Static
	//-------------
	// The following ensure buttons are big enough to press (min 44 pixels on iPhone 3G).
	private const float baseScreenHeight = 320.0f;
	private const float baseButtonHeight = 44.0f;



	// Public
	//-------
	public GameObject sphere;						// The sphere to move around with the accelerometer
	public float speed = 20.0f;						// Movement speed
	public float deadZone = 0.0f;					// Accelerometer deadzone. Keep this small, ideally less than 0.1.



	// Private
	//--------
	private float baseToCurrentScale = 1.0f;		// Scale base screen to current screen coordinates
	private Vector3 calibration = new Vector3(0.0f, 0.0f, -1.0f);	// Calibration values to save/load.
	private Rect border;							// World border to keep sphere on screen
	private float calibratedMsgTime;				// Timer to show calibrated message
	private Vector2 screenSize;						// Detect if device orientation changes
	private Vector3 sphereStartPosition;			// Start position of the sphere


	// Methods
	//--------

	/// <summary>
	/// Awake this instance.
	/// </summary>
	void Awake()
	{
		if (sphere != null)
		{
			sphereStartPosition = sphere.transform.position;
		}

		InitScreen();

		LoadSettings();
		FunkyCalibrate.RestoreCalibration(calibration);
		FunkyCalibrate.SetDeadZone(new Vector3(deadZone, deadZone, deadZone));
	}


	/// <summary>
	/// Raises the application quit event.
	/// </summary>
	void OnApplicationQuit()
	{
		SaveSettings();
	}


	/// <summary>
	/// Init info based on the screen resolution and orientation.
	/// </summary>
	void InitScreen()
	{
		screenSize = new Vector2(Screen.width, Screen.height);

		baseToCurrentScale = (float)Screen.height / baseScreenHeight;

		if ((sphere != null) && (Camera.main != null))
		{
			float distance = Vector3.Distance(Camera.main.transform.position, sphereStartPosition);
			Vector3 topLeft = Camera.main.ScreenToWorldPoint(new Vector3(0.0f, 0.0f, distance));
			Vector3 bottomRight = Camera.main.ScreenToWorldPoint(new Vector3(Screen.width, Screen.height, distance));
			border = new Rect(topLeft.x, topLeft.y, bottomRight.x - topLeft.x, bottomRight.y - topLeft.y);
		}
	}


	/// <summary>
	/// Loads the settings.
	/// </summary>
	void LoadSettings()
	{
		if ((PlayerPrefs.HasKey("calX")) && 
		    (PlayerPrefs.HasKey("calY")) && 
		    (PlayerPrefs.HasKey("calZ")))
		{
			calibration.x = Mathf.Clamp(PlayerPrefs.GetFloat("calX"), -1.0f, 1.0f);
			calibration.y = Mathf.Clamp(PlayerPrefs.GetFloat("calY"), -1.0f, 1.0f);
			calibration.z = Mathf.Clamp(PlayerPrefs.GetFloat("calZ"), -1.0f, 1.0f);
		}
	}


	/// <summary>
	/// Saves the settings.
	/// </summary>
	void SaveSettings()
	{
		PlayerPrefs.SetFloat("calX", calibration.x);
		PlayerPrefs.SetFloat("calY", calibration.y);
		PlayerPrefs.SetFloat("calZ", calibration.z);
		PlayerPrefs.Save();
	}


	/// <summary>
	/// Update is called once per frame.
	/// </summary>
	void Update()
	{
		float dt = Time.deltaTime;

		// Did the device orientation change?
		if ((screenSize.x != Screen.width) || (screenSize.y != Screen.height))
		{
			InitScreen();
		}

		// Use the accelerometer to move the sphere
		if (sphere != null)
		{
			Vector3 input = FunkyCalibrate.GetAccelerometerXY();

#if UNITY_EDITOR
			if (Input.GetKey(KeyCode.LeftArrow))
			{
				input.x = -1.0f;
			}
			else if (Input.GetKey(KeyCode.RightArrow))
			{
				input.x = 1.0f;
			}
			if (Input.GetKey(KeyCode.DownArrow))
			{
				input.y = -1.0f;
			}
			else if (Input.GetKey(KeyCode.UpArrow))
			{
				input.y = 1.0f;
			}
			input.Normalize();
#endif //UNITY_EDITOR

			Vector3 pos = sphere.transform.position + (input * speed * dt);

			pos.x = Mathf.Clamp(pos.x, border.xMin, border.xMax);
			pos.y = Mathf.Clamp(pos.y, border.yMin, border.yMax);

			sphere.transform.position = pos;
		}


		if (calibratedMsgTime > 0.0f)
		{
			calibratedMsgTime -= dt;
			if (calibratedMsgTime < 0.0f)
			{
				calibratedMsgTime = 0.0f;
			}
		}
	}


	/// <summary>
	/// Draw GUI.
	/// </summary>
	void OnGUI()
	{
		float addY = 25.0f;
		Vector2 pos = new Vector2(5, Screen.height - (addY * 3));
		Vector2 buttonSize = new Vector2(Mathf.Max(150.0f, 150.0f * baseToCurrentScale), Mathf.Max(baseButtonHeight, baseButtonHeight * baseToCurrentScale));

		// The calibration vector
		GUI.Label(new Rect(pos.x, pos.y, 1000, addY), "Calibration: " + calibration);
		pos.y += addY;

		// Accelerometer vector
		GUI.Label(new Rect(pos.x, pos.y, 1000, addY), "Accelerometer: " + Input.acceleration);
		pos.y += addY;

		// Calibrated accelerometer vector
		GUI.Label(new Rect(pos.x, pos.y, 1000, addY), "Calibrated Accelerometer: " + FunkyCalibrate.GetAccelerometer());
		pos.y += addY;


		// Button to calibrate the device
		pos = new Vector2(Screen.width - buttonSize.x - 5, 5);
		if (GUI.Button (new Rect(pos.x, pos.y, buttonSize.x, buttonSize.y), "Calibrate"))
		{
			calibration = FunkyCalibrate.Calibrate();
			calibratedMsgTime = 2.0f;
			SaveSettings();
		}
		pos.y += (5.0f + buttonSize.y);

		if (calibratedMsgTime > 0.0f)
		{
			GUI.Label(new Rect(pos.x, pos.y, 1000, addY), "CALIBRATED!");
			pos.y += addY;
		}
	}

}
